# 介绍

方案开发是一种使用类`JSON Schema`格式的预置数据，快速生成一个业务逻辑视图，并且能够被引用及复用。

## 使用场景

① 复杂后台管理项目中，某些相同的功能可能会多次出现在不同的使用场景中，并且**同时**具有**相似性**和**差异性**，而**组件化**不能更方便的解决类似的场景。

② 逻辑并不复杂，但是表单项非常多，需要反复编写大量**重复**的**布局代码**，导致**代码量大**，业务逻辑代码**维护不便**的场景。

③ 不同功能页面下，使用了相同的表单或表格等相同的功能模块，遇到**改动**时，需要重新在多个页面里多次**重复修改**代码的场景。

## 示例

某页面的详情功能，常规情形下，会出现大量重复的布局组件`el-row`、`el-col`及表单组件`el-form-item`等，进行取值展示时，也需要多次书写`form.xxx`。本是一个没有业务逻辑的展示型页面，但大量书写的重复的代码，导致行数过多，且如遇到增删某个展示项时，还需要重新调整布局代码。总体上来说，代码可读性低，可维护性差。

代码：

```html
<template>
  <el-form :model="form" size="small" label-width="100px">
    <el-row>
      <el-col :span="12">
        <el-form-item label="线路名称">
          {{ form.dispatchLine.lineName }}
        </el-form-item>
      </el-col>
      <el-col :span="12">
        <el-form-item label="出发站点">
          {{ form.dispatchLine.startStation ? form.dispatchLine.startStation.name : '' }}
        </el-form-item>
      </el-col>
    </el-row>
    <el-row>
      <el-col :span="12">
        <el-form-item label="目的站点">
          {{ form.dispatchLine.endStation ? form.dispatchLine.endStation.name : '' }}
        </el-form-item>
      </el-col>
      <el-col :span="12">
        <el-form-item label="客户名称">
          {{ form.customer ? form.customer.name : '' }}
        </el-form-item>
      </el-col>
    </el-row>
    <el-row>
      <el-col :span="12">
        <el-form-item label="线路类型">
          <template>
            <dictionary-select-name option-name="LINE_TYPE" :value="form.dispatchLine.lineType"></dictionary-select-name>
          </template>
        </el-form-item>
      </el-col>
      <el-col :span="12">
        <el-form-item label="出发城市">
          {{ form.dispatchLine.startStation.city.name }}
        </el-form-item>
      </el-col>
    </el-row>
    <el-row>
      <el-col :span="12">
        <el-form-item label="经停城市">
          {{ form.dispatchLine.stopOverCityName }}
        </el-form-item>
      </el-col>
      <el-col :span="12">
        <el-form-item label="目的城市">
          {{ form.dispatchLine.endStation.city.name }}
        </el-form-item>
      </el-col>
    </el-row>
  </el-form>
</template>

<script>
export default {
  data() {
    return {
      form: {
        dispatchLine: {
          startStation: {
            city: {},
          },
          endStation: {
            city: {},
          },
        },
        customer: {},
      },
    };
  },
}
</script>
```

仅视图层就有53行代码，改用Schema开发后可将视图层代码极致精简，可以使后端开发员修改前端代码时不需要关心界面布局及样式，只关心业务逻辑。

使用Schema开发后的代码，视图层1行，而配置项也仅需13行代码，并且无需关心布局代码和多层级数据取值容错：

```html
<template>
  <eagle-schema-form v-model="form" :schema="schema"></eagle-schema-form>
</template>

<script>
export default {
  data() {
    return {
      schema: {
        props: { size: 'small', 'label-width': '100px', span: 12 },
        items: [
          { prop: 'dispatchLine.lineName', label: '线路名称' },
          { prop: 'dispatchLine.startStation.name', label: '出发站点' },
          { prop: 'dispatchLine.endStation.name', label: '目的站点' },
          { prop: 'customer.name', label: '客户名称' },
          { is: 'dictionary-select-name', props: { 'option-name': 'LINE_TYPE' }, prop: 'customer.name', label: '线路类型' },
          { prop: 'dispatchLine.startStation.city.name', label: '出发城市' },
          { prop: 'dispatchLine.stopOverCityName', label: '经停城市' },
          { prop: 'dispatchLine.endStation.city.name', label: '经停城市' },
        ]
      },
      form: {
        dispatchLine: {
          startStation: {
            city: {},
          },
          endStation: {
            city: {},
          },
        },
        customer: {},
      },
    };
  },
}
</script>
```

也可以将Schema配置项单独保存在js文件中，需要的时候引入使用即可：

```js
// xxx.schema.js
export default {
  props: { size: 'small', 'label-width': '100px', span: 12 },
  items: [
    { prop: 'dispatchLine.lineName', label: '线路名称' },
    { prop: 'dispatchLine.startStation.name', label: '出发站点' },
    { prop: 'dispatchLine.endStation.name', label: '目的站点' },
    { prop: 'customer.name', label: '客户名称' },
    { is: 'dictionary-select-name', props: { 'option-name': 'LINE_TYPE' }, prop: 'customer.name', label: '线路类型' },
    { prop: 'dispatchLine.startStation.city.name', label: '出发城市' },
    { prop: 'dispatchLine.stopOverCityName', label: '经停城市' },
    { prop: 'dispatchLine.endStation.city.name', label: '经停城市' },
  ]
}
```

这样可以将业务逻辑结构和视图抽离并可复用。

```html
<template>
  <eagle-schema-form v-model="form" :schema="schema"></eagle-schema-form>
</template>

<script>
import xxxSchema from '@/schema/xxx.schema.js'l

export default {
  data() {
    return {
      schema: xxxSchema,
      form: {
        dispatchLine: {
          startStation: {
            city: {},
          },
          endStation: {
            city: {},
          },
        },
        customer: {},
      },
    };
  },
}
</script>
```

也可对原始配置项进行一个差异化的处理，比如配置项相同的情况下改变schema已配置的属性：

```html
<template>
  <eagle-schema-form v-model="form" :schema="schema"></eagle-schema-form>
</template>

<script>
import xxxSchema from '@/schema/xxx.schema.js'l

export default {
  data() {
    return {
      schema: { ...xxxSchema, props: { size: 'mini', 'label-width': '60px', span: 24 } },
      form: {
        dispatchLine: {
          startStation: {
            city: {},
          },
          endStation: {
            city: {},
          },
        },
        customer: {},
      },
    };
  },
}
</script>
```

或者在配置项的基础上再追加几个配置项：

```html
<template>
  <eagle-schema-form v-model="form" :schema="schema"></eagle-schema-form>
</template>

<script>
import xxxSchema from '@/schema/xxx.schema.js'l

export default {
  data() {
    return {
      schema: {
        ...xxxSchema,
        items: [
          ...xxxSchema.items,
          { prop: 'vehicleLicenseNum', label: '车牌号' },
          { prop: 'remark', label: '备注', spam: 24 },
        ]
      },
      form: {
        dispatchLine: {
          startStation: {
            city: {},
          },
          endStation: {
            city: {},
          },
        },
        customer: {},
      },
    };
  },
}
</script>
```

## 更多

更多具体示例，请查看`schema-form`、`schema-table`、`schema-page`文档说明。